const del = require('del');
const gulpZip = require('gulp-zip');
const log = require('fancy-log');
const fs = require('fs');
const cproc = require('child_process');
const path = require('path');
const gulpReplace = require('gulp-replace');
const git = require('gulp-git');

module.exports = function (gulp, config) {
  let version = {};

  let plugin_wp_block_version_re = /\*\w* Version:\W*(.*)/gm;
  let plugin_define_version_re = new RegExp('\'' + config.idCaps + '_VERSION\', \'(.*)\'', 'gm');
  let plugin_main_file = path.join(config.slug, config.slug + '.php');
  let plugin_readme_file = path.join(config.slug, 'README.md');

  if (config.mainFile === undefined) {
    config.mainFile = plugin_main_file;
  }

  if (config.readmeFile === undefined) {
    config.readmeFile = plugin_readme_file;
  }

  function getArgs () {
    return (argList => {
      let arg = {}, a, opt, thisOpt, curOpt;
      for (a = 0; a < argList.length; a++) {
        thisOpt = argList[ a ].trim();
        opt = thisOpt.replace(/^\-+/, '');

        if (opt === thisOpt) {
          // argument value
          if (curOpt) {
            arg[ curOpt ] = opt;
          }
          curOpt = null;
        } else {
          // argument name
          curOpt = opt;
          arg[ curOpt ] = true;
        }
      }
      return arg;
    })(process.argv);
  }

  function getVersion (fn) {
    let fileText = fs.readFileSync(fn);

    return {
      wpVersion: plugin_wp_block_version_re.exec(fileText)[ 1 ],
      defVersion: plugin_define_version_re.exec(fileText)[ 1 ],
    };
  }

  function escapeVersion (ver) {
    return ver.replace(/\./gm, '\\.');
  }

  function checkReadmeVersion (fn) {
    return new RegExp('### v' + escapeVersion(version.wpVersion) + '.*', 'gm').exec(fs.readFileSync(fn));
  }

  function checkReadmeReleaseDate (fn) {
    let t = new RegExp('### v' + escapeVersion(version.wpVersion) + '.*TBD.*', 'gm');
    return true; //!t.exec(fs.readFileSync(fn));
  }

  function withError (msg) {
    let err = new Error(msg);
    err.showStack = false;
    return err;
  }

  gulp.task('bump-version', done => {
    const args = getArgs();

    if (args.to === undefined) {
      done(withError('Usage: gulp bump-version --to NEWVER [--branch]'));
    }

    const inFiles = ['package.json', config.mainFile, config.readmeFile, path.join(config.slug, 'composer.json')];

    let stream = gulp.src(inFiles)
      .pipe(gulpReplace(/(\"version\": \")([\d\.]+-?\w*)(\")/gm, '$1' + args.to + '$3'))
      .pipe(gulpReplace(/(\* Version:\W+) ([\d\.]+-?\w*)/gm, '$1 ' + args.to))
      .pipe(gulpReplace(/(_VERSION\W+)\'([\d\.]+-?\w*)/gm, '$1\'' + args.to))
      .pipe(gulpReplace(/^## Change Log$/gm, '## Change Log\n\n### v' + args.to + ' - TBD'))
      .pipe(gulp.dest('.'));

    if (args.branch) {
      git.checkout('dev-' + args.to, { args: '-b' }, err => {
        if (err) throw err;
        stream = stream.pipe(git.add())
          .pipe(git.commit('Branch version bump'));
      });
    }

    return stream;
  });

  gulp.task('clean', done => {
    return del('./build/**/*');
  });

  gulp.task('check-version', done => {
    const args = getArgs();
    log(args.toString());
    done();
    version = getVersion(config.mainFile);

    if (version.isDev) {
      log.warn('******** DEVELOPMENT VERSION ********');
    }

    if (version.wpVersion !== version.defVersion) {
      done(withError('Version mismatch: WP Plugin Block says "' + version.wpVersion + '" but Plugin Defined version says "' + version.defVersion + '"'));
    }

    // Check for version changelog in readme and changelog
    if (!checkReadmeVersion(config.readmeFile)) {
      done(withError('Missing changelog entry in README.md'));
    }

    if (!checkReadmeReleaseDate(config.readmeFile) && !version.isDev) {
      done(withError('Missing release date for this version in README.md'));
    }

    done();
  });

  gulp.task('install-deps', done => {
    cproc.execSync('composer -q install --no-dev', {
      cwd: './build/' + config.slug + '/',
    });

    done();
  });

  gulp.task('dump-autoload', done => {
    cproc.execSync('composer -q dumpautoload', {
      cwd: './build/' + config.slug + '/',
    });

    done();
  });

  gulp.task('copy-plugin', done => {
    return gulp
      .src([
        path.join(config.slug, '**/*'),
        '!' + path.join(config.slug, 'node_modules/**/*'),
        '!' + path.join(config.slug, 'vendor/**/*'),
        //'!' + path.join(config.slug, 'assets/**/*'),
      ], { base: '.' })
      .pipe(gulp.dest('./build/'));
  });

  gulp.task('copy-assets', done => {
    return gulp
      .src('./build/assets/**/*')
      .pipe(gulp.dest('./build/' + config.slug + '/assets'));
  });

  gulp.task('copy-assets-dev', done => {
    const plugDir = './' + config.slug + '/assets/';
    let sources = [
      './build/assets/**/*',
      '!./build/assets/fonts/*',
      '!./build/assets/img/*',
    ];

    return gulp
      .src(sources)
      .pipe(gulp.dest(plugDir));
  });

  gulp.task('do-build', done => {
    let dest_file = config.slug + '-' + version.wpVersion + '.zip';

    log('Building version ' + version.wpVersion + ' to dist/' + dest_file + '...');

    return gulp
      .src(path.join('./build', config.slug, '**/*'), { base: './build' })
      .pipe(gulpZip(dest_file))
      .pipe(gulp.dest('dist/'));
  });

  async function doPublish (cb) {
    return new Promise((resolve, reject) => {
      let dest_file = './dist/' + config.slug + '-' + version.wpVersion + '.zip';

      cproc.execSync('ebsdev pluginrepo add-version ' + config.slug + ' ' + version.wpVersion + ' ' + dest_file, {
        cwd: './',
      });

      resolve();
    });
  }

  gulp.task('publish', gulp.series(['check-version', doPublish]));

  gulp.task('default', gulp.series('check-version', 'clean', 'copy-plugin', 'install-deps', 'do-build'));
};
