<?php

namespace EbsCommon\Wordpress;

use Carbon_Fields\Field\Field;

/**
 * Class TabBuilder
 * @package EbsCommon
 */
class TabBuilder {
	/**
	 * @var string Name of tab
	 */
	protected $name;
	/**
	 * @var array Tab items
	 */
	protected $items;
	/**
	 * @var int Tab sort order
	 */
	protected $order;

	/**
	 * TabBuilder constructor.
	 *
	 * @param     $name
	 * @param int $order
	 *
	 * @return $this
	 */
	public function __construct( $name, $order = 0 ) {
		$this->name  = $name;
		$this->items = [];
		$this->order = $order;

		return $this;
	}

	public static function make( $name, $order ): TabBuilder {
		return new self( $name, $order );
	}

	/**
	 * Add CarbonFields Field to collection
	 *
	 * Wraps Field::make()
	 *
	 * @param string $type
	 * @param string $name
	 * @param string $title
	 *
	 * @return Field
	 */
	public function addField( $type, $name, $title, $order = 0 ): Field {
		$newItem              = Field::make( $type, $name, $title );
		$this->items[ $name ] = [
			'name'  => $name,
			'order' => $order,
			'field' => $newItem,
		];

		$this->sortItems();

		return $newItem;
	}

	/**
	 * Sort items in TabBuilder
	 *
	 * @return void
	 */
	protected function sortItems(): void {
		usort( $this->items,
			function ( $a, $b ) {
				if ( $a['order'] == $b['order'] ) {
					return 0;
				}

				return $a['order'] < $b['order'] ? - 1 : 1;
			} );
	}

	/**
	 * Add CarbonFields Field to collection
	 *
	 * Wraps Field::make()
	 *
	 * @param string $type
	 * @param string $name
	 * @param string $title
	 *
	 * @return Field
	 */
	public function addField2( $type, $name, $title, $initFunc ): Field {
		$newItem = Field::make( $type, $name, $title );
		$initFunc( $newItem );

		$this->items[ $name ] = [
			'name'  => $name,
			'field' => $newItem,
		];

		$this->sortItems();

		return $newItem;
	}

	/**
	 * Get a Field from collection
	 *
	 * @param $name
	 *
	 * @return Field|null
	 */
	public function getField( $name ): ?Field {
		foreach ( $this->items as $item ) {
			if ( $item['name'] == $name ) {
				return $item['field'];
			}
		}

		return null;
	}

	/**
	 * Get array of defined items
	 *
	 * @return array
	 */
	public function getItemsArray(): array {
		return array_map( function ( $item ) {
			return $item['field'];
		},
			$this->items );
	}

	/**
	 * @return string
	 */
	public function getName(): string {
		return $this->name;
	}

	/**
	 * @return int
	 */
	public function getOrder(): int {
		return $this->order;
	}

	/**
	 * @return array
	 */
	public function getItems(): array {
		return $this->items;
	}
}
