<?php
/**
 * Trait definition to implement WordPress action/filter/shortcode hooking
 *
 * @link       https://everbearingservices.com
 * @since      1.0.0
 *
 * @package    EbsCommon
 * @subpackage EbsCommon/includes
 */

namespace EbsCommon\Wordpress\Traits;

trait WordpressHooksTrait {
	/**
	 * The array of actions registered with WordPress.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array $actions The actions registered with WordPress to fire when the plugin loads.
	 */
	protected $actions;
	/**
	 * The array of filters registered with WordPress.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      array $filters The filters registered with WordPress to fire when the plugin loads.
	 */
	protected $filters;

	/**
	 * The array of shortcodes to be registered with Wordpress
	 *
	 * @since   1.0.0
	 * @access  protected
	 * @var     array $shortcodes The shortcodes registered by the component
	 */
	protected $shortcodes;

	/**
	 * Add a new action to the collection to be registered with WordPress.
	 *
	 * @param string $hook          The name of the WordPress action that is being registered.
	 * @param string $callback      The name of the function definition on the $component.
	 * @param int    $priority      Optional. The priority at which the function should be fired. Default is 10.
	 * @param int    $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1.
	 *
	 * @since    1.0.0
	 *
	 */
	public function add_action( $hook, $callback, $priority = 10, $accepted_args = 1 ) {
		if ( is_null( $this->actions ) ) {
			$this->actions = [];
		}

		$this->actions = $this->add( $this->actions, $hook, $this, $callback, $priority, $accepted_args );
	}

	/**
	 * A utility function that is used to register the actions and hooks into a single
	 * collection.
	 *
	 * @param array  $hooks         The collection of hooks that is being registered (that is, actions or filters).
	 * @param string $hook          The name of the WordPress filter that is being registered.
	 * @param object $component     A reference to the instance of the object on which the filter is defined.
	 * @param string $callback      The name of the function definition on the $component.
	 * @param int    $priority      The priority at which the function should be fired.
	 * @param int    $accepted_args The number of arguments that should be passed to the $callback.
	 *
	 * @return   array                                  The collection of actions and filters registered with WordPress.
	 * @since    1.0.0
	 * @access   private
	 *
	 */
	private function add( $hooks, $hook, $component, $callback, $priority, $accepted_args ) {

		$hooks[] = [
			'hook'          => $hook,
			'component'     => $component,
			'callback'      => $callback,
			'priority'      => $priority,
			'accepted_args' => $accepted_args,
		];

		return $hooks;

	}

	/**
	 * Add a new filter to the collection to be registered with WordPress.
	 *
	 * @param string $hook          The name of the WordPress filter that is being registered.
	 * @param string $callback      The name of the function definition on the $component.
	 * @param int    $priority      Optional. The priority at which the function should be fired. Default is 10.
	 * @param int    $accepted_args Optional. The number of arguments that should be passed to the $callback. Default is 1
	 *
	 * @since    1.0.0
	 */
	public function add_filter( $hook, $callback, $priority = 10, $accepted_args = 1 ) {
		if ( is_null( $this->filters ) ) {
			$this->filters = [];
		}

		$this->filters = $this->add( $this->filters, $hook, $this, $callback, $priority, $accepted_args );
	}

	/**
	 * Add a new shortcode to the collection to be registered with WordPress.
	 *
	 * @params   string $shortcode_name Shortcode name to register
	 * @params   callable $callback Function to register as shortcode
	 *
	 * @since    1.0.0
	 */
	public function add_shortcode( $shortcode_name, $callback ) {
		if ( is_null( $this->shortcodes ) ) {
			$this->shortcodes = [];
		}

		$this->shortcodes[] = [
			'name'     => $shortcode_name,
			'callback' => [ $this, $callback ],
		];
	}

	/**
	 * Register the filters and actions with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function register_hooks() {
		if ( ! is_null( $this->filters ) ) {
			foreach ( $this->filters as $hook ) {
				add_filter( $hook['hook'],
					[
						$hook['component'],
						$hook['callback'],
					],
					$hook['priority'],
					$hook['accepted_args'] );
			}
		}

		if ( ! is_null( $this->actions ) ) {
			foreach ( $this->actions as $hook ) {
				add_action( $hook['hook'],
					[
						$hook['component'],
						$hook['callback'],
					],
					$hook['priority'],
					$hook['accepted_args'] );
			}
		}

		if ( ! is_null( $this->shortcodes ) ) {
			foreach ( $this->shortcodes as $shortcode ) {
				add_shortcode(
					$shortcode['name'],
					$shortcode['callback']
				);
			}
		}
	}

}