<?php

namespace EbsCommon\Wordpress;

use Carbon_Fields\Field\Field;

/**
 * Class TabBuilder
 * @package EbsCommon
 */
class TabBuilder {
	/**
	 * @var string Name of tab
	 */
	protected $name;
	/**
	 * @var array Tab items
	 */
	protected $items;
	/**
	 * @var int Tab sort order
	 */
	protected $order;

	/**
	 * TabBuilder constructor.
	 *
	 * @param     $name
	 * @param int $order
	 *
	 * @return $this
	 */
	public function __construct( $name, $order = 0 ) {
		$this->name  = $name;
		$this->items = [];
		$this->order = $order;

		return $this;
	}

	public static function make( $name, $order ) {
		return new self( $name, $order );
	}

	/**
	 * Add CarbonFields Field to collection
	 *
	 * Wraps Field::make()
	 *
	 * @param string $type
	 * @param string $name
	 * @param string $title
	 *
	 * @return Field
	 */
	public function addField( $type, $name, $title, $order = 0 ) {
		$newItem              = Field::make( $type, $name, $title );
		$this->items[ $name ] = [
			'name'  => $name,
			'order' => $order,
			'field' => $newItem,
		];

		$this->sortItems();

		return $newItem;
	}

	/**
	 * Add CarbonFields Field to collection
	 *
	 * Wraps Field::make()
	 *
	 * @param string $type
	 * @param string $name
	 * @param string $title
	 *
	 * @return Field
	 */
	public function addField2( $type, $name, $title, $initFunc ) {
		$newItem              = Field::make( $type, $name, $title );
		$initFunc($newItem);

		$this->items[ $name ] = [
			'name'  => $name,
			'field' => $newItem,
		];

		$this->sortItems();

		return $newItem;
	}


	protected function sortItems() {
		usort( $this->items,
			function ( $a, $b ) {
				return $a['order'] > $b['order'];
			} );
	}

	/**
	 * Get a Field from collection
	 *
	 * @param $name
	 *
	 * @return Field
	 */
	public function getField( $name ) {
		foreach ($this->items as $item) {
			if ($item['name'] == $name) {
				return $item['field'];
			}
		}

		return null;
	}

	public function getItemsArray() {
		return array_map( function ( $item ) {
			return $item['field'];
		},
			$this->items );
	}

	/**
	 * @return string
	 */
	public function getName(): string {
		return $this->name;
	}

	/**
	 * @return int
	 */
	public function getOrder(): int {
		return $this->order;
	}

	/**
	 * @return array
	 */
	public function getItems(): array {
		return $this->items;
	}
}
