<?php

namespace EbsCommon\Wordpress;

use Carbon_Fields\Container\Container;
use EbsCommon\Wordpress\Traits\WordpressHooksTrait;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use function EbsCommon\joinPaths;

/**
 *
 */
define( 'EBS_COMMON_DEV_MODE', strpos( get_home_url(), 'dev.everbearingservices.com' ) !== false );

/**
 * Class Plugin
 * @package EbsCommon\Wordpress
 */
class Plugin {
	use WordpressHooksTrait;

	public static $log_handlers = [];
	public static $form_handler_options = [
		'action'   => '',
		'callback' => '',
	];
	protected static $loggers = [];
	/**
	 * @var array
	 */
	public $admin_notices;

	/**
	 * @var
	 */
	protected $name;
	/**
	 * @var
	 */
	protected $version;
	/**
	 * @var
	 */
	protected $plugin_file;
	/**
	 * @var string
	 */
	protected $plugin_path;
	protected $form_handlers;

	/** @var array List of directories used by plugin */
	protected $plugin_dirs;

	/**
	 * @var
	 */
	protected $admin;

	protected $adminMenus;

	/**
	 * @var Updater
	 */
	protected $updater;

	/**
	 * Plugin constructor.
	 *
	 * @param $name
	 * @param $version
	 * @param $file
	 */
	public function __construct( $name, $version, $file ) {
		$this->name          = $name;
		$this->version       = $version;
		$this->form_handlers = [];
		$this->plugin_file   = $file;
		$this->plugin_path   = plugin_dir_path( $this->plugin_file );
		$this->admin_notices = [];

		$this->addPluginDir( 'plugin', plugin_dir_path( $this->plugin_file ), false )
		     ->addPluginDir( 'data', wp_get_upload_dir()['basedir'] . '/' . $this->name, true )
		     ->addPluginDir( 'cache_base', ABSPATH . wp_basename( content_url() ) . '/cache/', true )
		     ->addPluginSubdir( 'cache_base', 'cache', $this->name, true );

		$this->adminMenus = new AdminMenus( $this, 'ADMIN_MENU', 'ADMIN_MENU_SLUG' );
		$this->adminMenus->register_hooks();

		$this->updater = new Updater( $this );
	}

	/**
	 * Add a PluginDir entry
	 *
	 * @param string $name              Name of PluginDir entry
	 * @param string $path              Path to PluginDir entry
	 * @param bool   $create_if_missing Should the directory be created if missing?  Default true
	 *
	 * @return $this
	 */
	public function addPluginDir( $name, $path, $create_if_missing = true ) {
		$this->plugin_dirs[ $name ] = [
			'path'              => $path,
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	/**
	 * Add a subdirectory to another plugin directory
	 *
	 * @param string $parent            Parent PluginDir to add subdirectory to
	 * @param string $name              Name of subdirectory PluginDir entry
	 * @param string $path              Allows specification of physical subdirectory name, defaults ot $name
	 * @param bool   $create_if_missing Should directory be created if missing?  Default true
	 *
	 * @return $this
	 * @throws ArgumentError
	 *
	 */
	public function addPluginSubdir( $parent, $name, $path = '', $create_if_missing = true ) {
		if ( ! array_key_exists( $parent, $this->plugin_dirs ) ) {
			throw new ArgumentError( "Parent dir {$parent} does not exist" );
		}

		$pd                         = $this->plugin_dirs[ $parent ];
		$this->plugin_dirs[ $name ] = [
			'path'              => joinPaths( $pd['path'], $path ? $path : $name ),
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	/**
	 * @param $options
	 *
	 * @return $this
	 */
	public function addFormHandler( $options ) {
		$this->form_handlers[] = array_merge( [], self::$form_handler_options, $options );

		return $this;
	}

	public function wp_loaded() {
		$this->process_form();

//		do_action($this->getPluginSlug(''))
	}

	public function process_form() {
		if ( $_SERVER['REQUEST_METHOD'] != 'POST' ) {
			return;
		}

		if ( isset( $_REQUEST['action'] ) ) {
			$action = $_REQUEST['action'];
		} else {
			return;
		}

		foreach ( $this->form_handlers as $form_handler ) {

			if ( $form_handler['action'] == $action ) {
				// Process form
				call_user_func( $form_handler['callback'] );

				$redirect_url = $_SERVER['REQUEST_URI'];
				if ( isset( $form_handler['redirect'] ) && $form_handler['redirect'] ) {
					$redirect_url = $form_handler['redirect'];
				}

				wp_redirect( $redirect_url );
				exit;
			}
		}
	}

	/**
	 * Get absolute
	 *
	 * @param $path
	 *
	 * @return string
	 */
	public function getAssetUrl( $path ) {
		return plugin_dir_url( $this->plugin_file ) . 'assets/' . $path;
	}

	public function getAdminMenuSlug() {
		return $this->adminMenus->getMenuSlug();
	}

	public function getAdminMenus() {
		return $this->adminMenus;
	}

	/**
	 * @return mixed
	 */
	public function getVersion() {
		return $this->version;
	}

	/**
	 * @return mixed
	 */
	public function getPluginFile() {
		return $this->plugin_file;
	}

	/**
	 * @param $adminbar
	 *
	 * @return mixed
	 */
	public function add_dev_adminbar_indicator( $adminbar ) {
		$adminbar->add_node( [
			'id'    => 'ebs-common',
			'title' => "DEV v{$this->version}",
			'href=' => '#',
		] );

		return $adminbar;
	}

	/**
	 *
	 */
	public function add_dev_adminbar_css() {
		echo '<style type="text/css">' .
		     TemplateLoader::render( 'adminbar-inline-css.css', [] ) .
		     '</style>';
	}

	/**
	 * @param array $items
	 *
	 * @return array
	 */
	public function registerUpdateCheck( $items = [] ) {
		$items[] = [
			'basename' => $this->getName() . '/' . basename( $this->plugin_file ),
			'name'     => $this->getName(),
			'url'      => "https://ebsdev.everbearingservices.com/api/v1/plugin-repos/{$this->getName()}/latest",
			'version'  => $this->version,
		];

		return $items;
	}

	/**
	 * @return mixed
	 */
	public function getName() {
		return $this->name;
	}

	public function registerAdminMenuItems( $menu_items ) {
		if ( $this->enableDashboard() ) {
			$menu_items[] = [
				'title'  => 'Dashboard',
				'slug'   => 'dashboard',
				'func',
				[ $this, 'render_dashboard_page' ],
				'parent' => $this->adminMenus->getMenuSlug(),
			];
		}

		return $menu_items;
	}

	/**
	 * Should the Dasbhoard module be enabled
	 *
	 * @return bool
	 */
	public function enableDashboard() {
		return true;
	}

	/**
	 * Specify position of Top llevel menu in WP Admin menu sidebar
	 *
	 * @return int Top level menu position
	 */
	public function topLevelMenuOrder() {
		return 2;
	}

	/**
	 * @return int
	 */
	public function dashboardMenuOrder() {
		return - 1;
	}

	/**
	 * Sort order for Settings menu item
	 *
	 * @return int Relative order of Settings
	 */
	public function settingsMenuOrder() {
		return 99999;
	}

	/**
	 * @return bool
	 */
	public function enableSettings() {
		return true;
	}

	/**
	 * @param $level
	 * @param $args
	 */
	public function addAdminNotice( $level, $args ) {
		$args['level'] = $level;

		$this->admin_notices[] = $args;
	}

	/**
	 * Add flash notice for a User
	 *
	 * Defaults to current user
	 *
	 * @param string $level   Notice level, [warning|info|success|error]
	 * @param array  $args    Flash notice arguments
	 * @param int    $user_id User ID to add message for
	 *
	 * @return $this
	 */
	public function addFlashNotice( $level, $args, $user_id = null ) {
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}
		$args['level'] = $level;

		update_user_meta( $user_id, $this->getPluginSlug( 'flash_message_' . microtime() ), $args );

		return $this;
	}

	/**
	 * @param        $slug
	 * @param string $sep
	 *
	 * @return string
	 */
	public function getPluginSlug( $slug, $sep = '-' ) {
		return $this->name . $sep . $slug;
	}

	/**
	 *
	 */
	public function renderNotices() {
		$ns = array_map( function ( $x ) {
			return $this->configNotice( $x );
		},
			$this->admin_notices );

		$flash_notices = [];
		$user          = get_current_user_id();
		$metadata      = get_user_meta( $user );

		$slug = $this->getPluginSlug( 'flash_message' );
		foreach ( $metadata as $key => $value ) {
			$t = substr( $key, 0, strlen( $slug ) ) == $slug;
			if ( $t ) {
				$flash_notices[] = $this->configNotice( unserialize( $value[0] ) );
				delete_user_meta( $user, $key );
			}
		}

		echo TemplateLoader::render( 'notices.html.twig',
			[
				'plugin_slug'   => $this->getName(),
				'admin_notices' => $ns,
				'flash_notices' => $flash_notices,
			] );
	}

	/**
	 * @param $n
	 *
	 * @return mixed
	 */
	public function configNotice( $n ) {
		if ( ! isset( $n['content'] ) ) {
			$n['content'] = '';
		}

		if ( isset( $n['text'] ) ) {
			$n['content'] = '<p>' . $n['text'] . '</p>';
		}

		if ( isset( $n['cb'] ) ) {
			if ( is_callable( $n['cb'] ) ) {
				$n['content'] = $n['cb']();
			}
		}

		return $n;
	}

	/**
	 * @param SettingsBuilder $builder
	 */
	public function addSettings( $builder ) {
		$tab = $builder->addTab( 'Modules', 10 );
		$tab->addField( 'html', 'crb_settings_label_1', '' )
		    ->set_html( '<h2>Modules</h2>' );
		$tab->addField( 'checkbox', 'crb_settings_enable_faqs', 'Enable FAQs' );
		$tab->addField( 'checkbox', 'crb_settings_enable_masonry', 'Enable Masonry?' );

	}

	/**
	 *
	 */
	public function registerCarbonFields() {
		$this->createSettingsContainer();

		do_action( $this->getPluginSlug( 'register_fields' ) );
	}

	protected function createSettingsContainer() {
		$container = Container::make( 'theme_options', 'Settings' )
		                      ->set_page_parent( $this->adminMenus->getMenuSlug() );

		$builder = SettingsBuilder::make();

		do_action( $this->getPluginSlug( 'register_settings' ), $builder );

		$builder->apply( $container );
	}

	/**
	 *
	 */
	public function loadCarbonFields() {
		\Carbon_Fields\Carbon_Fields::boot();
	}

	/**
	 *
	 */
	public function startSession() {
		if ( ! session_id() && ! headers_sent() ) {
			session_start();
		}
	}

	/**
	 *
	 */
	public function endSession() {
		session_destroy();
	}

	/**
	 *
	 */
	public function run() {
		add_action( 'after_setup_theme', [ $this, 'loadCarbonFields', ] );

		add_action( 'admin_notices', [ $this, 'renderNotices' ] );

		add_action( 'carbon_fields_register_fields', [ $this, 'registerCarbonFields' ] );

//		add_action( 'init', [ $this, 'init' ] );


		$this->init();
	}

	/**
	 * Init
	 *
	 */
	public function init() {
		add_action( 'init', [ $this, 'wp_init' ] );
		add_action( 'after_setup_theme', [ $this, 'startSession' ], 0 );
		add_action( 'wp_logout', [ $this, 'endSession' ] );
		add_action( 'wp_login', [ $this, 'endSession' ] );

		if ( EBS_COMMON_DEV_MODE ) {
			$this->add_action( 'admin_bar_menu', 'add_dev_adminbar_indicator', 3, 1 );
			$this->add_action( 'wp_head', 'add_dev_adminbar_css', 100 );
			$this->add_action( 'admin_head', 'add_dev_adminbar_css', 100 );
		}

		$this->add_filter( 'register_admin_menu_items', 'registerAdminMenuItems', 10, 1 );
		$this->add_filter( 'admin_menu_order_dashboard', 'dashboardMenuOrder' );
		$this->add_filter( 'admin_menu_order_crb_carbon_fields_container_settings.php', 'settingsMenuOrder' );
		$this->add_filter( $this->getPluginSlug( 'register_settings' ), 'addSettings' );

		$this->add_filter( $this->getPluginSlug( 'update_plugins' ), 'registerUpdateCheck' );
		$this->add_action( 'wp_loaded', 'wp_loaded' );

		TemplateLoader::addTemplatePath( $this->plugin_path . 'templates' );
		TemplateLoader::setContext( [
			'PLUGIN' => [
				'slug'     => $this->name,
				'base_url' => plugin_dir_url( $this->plugin_file ),
			],
		] );

		$this->register_hooks();


		do_action( $this->getPluginSlug( 'init' ) );
	}

	/**
	 * [ACTION] Stuff to do on WP init
	 *
	 * @see init
	 */
	public function wp_init() {
		$this->createPluginDirs();
	}

	protected function createPluginDirs() {
		foreach ( $this->plugin_dirs as $dir ) {
			if ( $dir['create_if_missing'] && ! file_exists( $dir['path'] ) ) {
				self::get_logger()
				    ->debug( "PluginDir '{$dir['path']}' does not exist, creating" );
				mkdir( $dir['path'] );
			}
		}
	}

	public function get_logger( $name = null ) {
		if ( is_null( $name ) ) {
			$name = $this->getName();
		}

		if ( array_key_exists( $name, self::$loggers ) ) {
			return self::$loggers[ $name ];
		}
		if ( empty( self::$log_handlers ) ) {
			try {
				self::$log_handlers[] = new StreamHandler( $this->getPluginDir( 'data' ) . '/' . $this->name . '_log.txt' );
			} catch ( \Exception $e ) {
				error_log( "Failed to open logfile: {$e->getMessage()}" );
			}
		}
		self::$loggers[ $name ] = new Logger( $name );
		foreach ( self::$log_handlers as $handler ) {
			self::$loggers[ $name ]->pushHandler( $handler, Logger::DEBUG );
		}

		return self::$loggers[ $name ];
	}

	/**
	 * Retrieve a PluginDir
	 *
	 * @param string $name
	 *
	 * @return string|null
	 */
	public function getPluginDir( $name ) {
		if ( ! array_key_exists( $name, $this->plugin_dirs ) ) {
			return null;
		}

		return $this->plugin_dirs[ $name ]['path'];
	}

}