<?php

namespace EbsCommon\Wordpress\Modules;

/**
 * FAQs module
 *
 * @link       https://everbearingservices.com
 * @since      1.0.0
 *
 * @package    EbsCommon
 * @subpackage EbsCommon/includes/components/public
 */

use Carbon_Fields\Field;
use EbsCommon\Wordpress\Component;
use EbsCommon\Wordpress\Traits\WordpressHooksTrait;

/**
 * FAQs module
 *
 * @package    EbsCommon
 * @author     Everbearing Services <info@everbearingservices.com>
 */
class FaqsComponent extends Component {
	use WordpressHooksTrait;

	protected $faqs;

	public function __construct( $plugin ) {
		parent::__construct( $plugin );
	}

	public function register_fields() {
		if ( get_option( '_crb_settings_enable_faqs' ) != 'yes' ) {
			return null;
		}

		$t = $this->create_theme_option_container( 'FAQs', $this->plugin->getAdminMenus()->getMenuSlug() )
		          ->add_fields( [
			          Field::make( 'complex', 'crb_questions', __( 'Frequently Asked Questions', 'ebs-common' ) )
			               ->set_layout( 'tabbed-vertical' )
			               ->add_fields( [
				               Field::make( 'text', 'question', 'Question' )
				                    ->set_required( true )
				                    ->set_help_text( 'Titles do not need to be UPCASED.  This is done automatically when displayed' ),
				               Field::make( 'text', 'slug', 'Slug' )
				                    ->set_required( true )
				                    ->set_help_text( 'This is used to generate named Anchor tag to allow # addressing in URLs' ),
				               Field::make( 'rich_text', 'answer', 'Answer' )
				                    ->set_required( true )
				                    ->set_help_text( 'Use HTML tags for formatting' ),
			               ] )
			               ->set_help_text( 'Tab order determines display order' )
			               ->set_header_template( '
			              <% if (question) { %>
			                <%- question %>
			                <% } %>
			              ' ),
		          ] );
	}

	public function render_faqs( $atts ) {
		wp_enqueue_script( $this->get_name() );
		wp_enqueue_style( $this->get_name() );

		$atts = shortcode_atts( [
			'format' => 'columns',
		],
			$atts,
			'faqs' );

		if ( is_null( $this->faqs ) ) {
			$this->faqs = carbon_get_theme_option( 'crb_questions' );
		}

		foreach ( $this->faqs as &$faq ) {
			$faq['answer'] = apply_filters( 'the_content', $faq['answer'] );
		}

		$plugin = ebs_common_get_plugin();

		return $plugin->render_template( 'shortcodes/faqs.html.twig',
			[
				'format' => $atts['format'],
				'faqs'   => $this->faqs,
			] );
	}

	public function ebs_init() {
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function register_scripts() {
		wp_register_script( $this->plugin->getPluginSlug('faqs'),
			plugin_dir_url( __FILE__ ) . 'ebs-common-faqs.js',
			[ 'jquery' ],
			$this->plugin->getVersion(),
			true );

		wp_register_style(  $this->plugin->getPluginSlug('faqs'),
			plugin_dir_url( __FILE__ ) . 'ebs-common-faqs.css',
			[],
			$this->plugin->getVersion(),
			'all' );
	}

	/**
	 *
	 * Module initialization
	 *
	 * @since 1.0.0
	 */
	protected function init() {
		$this->add_action( 'wp_enqueue_scripts', 'register_scripts' );
		$this->add_filter( 'carbon_fields_register_fields', 'register_fields' );
		$this->add_shortcode( 'faqs', 'render_faqs' );
	}

}
