<?php

namespace EbsCommon\Wordpress;

use Carbon_Fields\Carbon_Fields;
use Carbon_Fields\Container;
use EbsCommon\Wordpress\Modules\FaqsComponent;
use EbsCommon\Wordpress\Modules\SchemaComponent;
use Exception;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use WP_Admin_Bar;
use function EbsCommon\joinPaths;

/**
 *
 */
define( 'EBS_COMMON_DEV_MODE', strpos( get_home_url(), 'dev.everbearingservices.com' ) !== false );

/**
 * Class Plugin
 * @package EbsCommon\Wordpress
 */
class Plugin {
	public static $log_handlers = [];

	protected static $loggers = [];

	/**
	 * @var array
	 */
	public $admin_notices;

	/**
	 * @var
	 */
	protected $name;
	/**
	 * @var
	 */
	protected $version;

	/**
	 * @var
	 */
	protected $plugin_file;

	/**
	 * @var string
	 */
	protected $plugin_path;

	/** @var array List of directories used by plugin */
	protected $plugin_dirs;

	/**
	 * @var
	 */
	protected $admin;

	protected $adminMenus;

	protected $components;

	/**
	 * @var Updater
	 */
	protected $updater;

	/**
	 * @var FilesystemAdapter Symfony Cache object
	 */
	protected $cache;

	/**
	 * Plugin constructor.
	 *
	 * @param $name
	 * @param $version
	 * @param $file
	 *
	 * @throws ArgumentError
	 */
	public function __construct( $name, $version, $file ) {
		$this->name          = $name;
		$this->version       = $version;
		$this->plugin_file   = $file;
		$this->plugin_path   = plugin_dir_path( $this->plugin_file );
		$this->admin_notices = [];

		$this->components = [];

		$this->registerComponent('adminMenus', AdminMenus::class, []);
		$this->registerComponent('updater', Updater::class, []);

		if ( $this->enableSchemaModule() ) {
			$this->registerComponent( 'schema', SchemaComponent::class, [] );
		}

		if ( $this->enableFaqsModule() ) {
			$this->registerComponent( 'faqs', FaqsComponent::class, [] );
		}

		$this->addPluginDir( 'plugin', plugin_dir_path( $this->plugin_file ), false )
		     ->addPluginDir( 'data', wp_get_upload_dir()['basedir'] . '/' . $this->name, true )
		     ->addPluginDir( 'cache_base', ABSPATH . wp_basename( content_url() ) . '/cache/', true )
		     ->addPluginSubdir( 'cache_base', 'cache', $this->name, true );


		$this->cache = new FilesystemAdapter( 'cache', 0, ABSPATH . wp_basename( content_url() ) . '/cache/' . $name );
	}

	protected function enableSchemaModule() {
		return true;
	}

	public function registerComponent( $name, $class, $deps ) {
		$this->components[ $name ] = [
			'class'    => $class,
			'deps'     => $deps,
			'instance' => new $class( $this ),
		];
	}

	public function getComponent($name) {
		if (isset($this->components[$name])) {
			return $this->components[$name]['instance'];
		}

		return null;
	}

	protected function enableFaqsModule() {
		return true;
	}

	/**
	 * Add a subdirectory to another plugin directory
	 *
	 * @param string $parent            Parent PluginDir to add subdirectory to
	 * @param string $name              Name of subdirectory PluginDir entry
	 * @param string $path              Allows specification of physical subdirectory name, defaults ot $name
	 * @param bool   $create_if_missing Should directory be created if missing?  Default true
	 *
	 * @return $this
	 * @throws ArgumentError
	 *
	 */
	public function addPluginSubdir( $parent, $name, $path = '', $create_if_missing = true ) {
		if ( ! array_key_exists( $parent, $this->plugin_dirs ) ) {
			throw new ArgumentError( "Parent dir {$parent} does not exist" );
		}

		$pd                         = $this->plugin_dirs[ $parent ];
		$this->plugin_dirs[ $name ] = [
			'path'              => joinPaths( $pd['path'], $path ? $path : $name ),
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	/**
	 * Add a PluginDir entry
	 *
	 * @param string $name              Name of PluginDir entry
	 * @param string $path              Path to PluginDir entry
	 * @param bool   $create_if_missing Should the directory be created if missing?  Default true
	 *
	 * @return $this
	 */
	public function addPluginDir( $name, $path, $create_if_missing = true ) {
		$this->plugin_dirs[ $name ] = [
			'path'              => $path,
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	public function cache() {
		return $this->cache;
	}

	/**
	 * Get absolute
	 *
	 * @param $path
	 *
	 * @return string
	 */
	public function getAssetUrl( $path ) {
		return plugin_dir_url( $this->plugin_file ) . 'assets/' . $path;
	}

	/**
	 * @return mixed
	 */
	public function getPluginFile() {
		return $this->plugin_file;
	}

	/**
	 * @param WP_Admin_Bar $adminbar
	 *
	 * @return mixed
	 */
	public function add_dev_adminbar_indicator( $adminbar ) {
		$adminbar->add_node( [
			'id'    => 'ebs-common',
			'title' => "DEV v{$this->version}",
			'href=' => '#',
		] );

		$adminbar->add_node( [
			'id'     => 'ebs-common-test',
			'parent' => 'ebs-common',
			'title'  => 'TEST',
			'href'   => '#',
		] );

		return $adminbar;
	}

	/**
	 * @param array $items
	 *
	 * @return array
	 */
	public function registerUpdateCheck( $items = [] ) {
		$items[] = [
			'basename' => $this->getUpdateBasename(),
			'name'     => $this->getName(),
			'url'      => $this->getUpdateUrl(),
			'version'  => $this->getVersion(),
		];

		return $items;
	}

	public function getUpdateBasename() {
		return $this->getName() . '/' . basename( $this->plugin_file );
	}

	/**
	 * @return mixed
	 */
	public function getName() {
		return $this->name;
	}

	public function getUpdateUrl() {
		return "https://ebsdev.everbearingservices.com/api/v1/plugin-repos/{$this->getName()}/latest";
	}

	/**
	 * @return mixed
	 */
	public function getVersion() {
		return $this->version;
	}

	public function registerAdminMenuItems( $menu_items ) {
		if ( $this->enableDashboard() ) {
			$menu_items[] = [
				'title'  => 'Dashboard',
				'slug'   => 'dashboard',
				'func',
				[ $this, 'render_dashboard_page' ],
				'parent' => $this->getComponent('adminMenus')->getMenuSlug(),
			];
		}

		return $menu_items;
	}

	/**
	 * Should the Dasbhoard module be enabled
	 *
	 * @return bool
	 */
	public function enableDashboard() {
		return true;
	}

	/**
	 * Specify position of Top llevel menu in WP Admin menu sidebar
	 *
	 * @return int Top level menu position
	 */
	public function topLevelMenuOrder() {
		return 2;
	}

	/**
	 * @return int
	 */
	public function dashboardMenuOrder() {
		return - 1;
	}

	/**
	 * Sort order for Settings menu item
	 *
	 * @return int Relative order of Settings
	 */
	public function settingsMenuOrder() {
		return 99999;
	}

	/**
	 * @return bool
	 */
	public function enableSettings() {
		return true;
	}

	/**
	 * @param $level
	 * @param $args
	 */
	public function addAdminNotice( $level, $args ) {
		$args['level'] = $level;

		$this->admin_notices[] = $args;
	}

	/**
	 * Add flash notice for a User
	 *
	 * Defaults to current user
	 *
	 * @param string $level   Notice level, [warning|info|success|error]
	 * @param array  $args    Flash notice arguments
	 * @param int    $user_id User ID to add message for
	 *
	 * @return $this
	 */
	public function addFlashNotice( $level, $args, $user_id = null ) {
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}
		$args['level'] = $level;

		update_user_meta( $user_id, $this->getPluginSlug( 'flash_message_' . microtime() ), $args );

		return $this;
	}

	/**
	 * @param        $slug
	 * @param string $sep
	 *
	 * @return string
	 */
	public function getPluginSlug( $slug, $sep = '-' ) {
		return $this->name . $sep . $slug;
	}

	/**
	 *
	 */
	public function renderNotices() {
		$ns = array_map( function ( $x ) {
			return $this->configNotice( $x );
		},
			$this->admin_notices );

		$flash_notices = [];
		$user          = get_current_user_id();
		$metadata      = get_user_meta( $user );

		$slug = $this->getPluginSlug( 'flash_message' );
		foreach ( $metadata as $key => $value ) {
			$t = substr( $key, 0, strlen( $slug ) ) == $slug;
			if ( $t ) {
				$flash_notices[] = $this->configNotice( unserialize( $value[0] ) );
				delete_user_meta( $user, $key );
			}
		}

		echo TemplateLoader::render( 'notices.html.twig',
			[
				'plugin_slug'   => $this->getName(),
				'admin_notices' => $ns,
				'flash_notices' => $flash_notices,
			] );
	}

	/**
	 * @param $n
	 *
	 * @return mixed
	 */
	public function configNotice( $n ) {
		if ( ! isset( $n['content'] ) ) {
			$n['content'] = '';
		}

		if ( isset( $n['text'] ) ) {
			$n['content'] = '<p>' . $n['text'] . '</p>';
		}

		if ( isset( $n['cb'] ) ) {
			if ( is_callable( $n['cb'] ) ) {
				$n['content'] = $n['cb']();
			}
		}

		return $n;
	}

	/**
	 * @param SettingsBuilder $builder
	 */
	public function addSettings( $builder ) {
		$this->registerGeneralSettings( $builder );
		$this->registerEmailSettings( $builder );
	}

	protected function registerGeneralSettings( $builder ) {
		$tab = $builder->addTab( 'General', - 10 );
		$tab->addField( 'text', 'crb_password_strength', __( 'Password Strength', 'lpo-custom' ) )
		    ->set_default_value( 2 )
		    ->set_required( true );
		$tab->addField( 'textarea', 'crb_password_hint', __( 'Password hint', 'lpo-custom' ) )
		    ->set_default_value( 'Hint: The password should be at least twelve characters long. To make it stronger, use upper and lower case letters, numbers, and symbols like ! " ? $ % ^ &amp; ).' )
		    ->set_required( true );
		$tab->addField( 'checkbox', 'crb_enable_maintenance_mode', __( 'Enable maintenance mode' ) )
		    ->set_default_value( false );

	}

	protected function registerEmailSettings( $builder ) {
		$tab = $builder->addTab( 'Email', 8 );
		$tab->addField( 'text', 'crb_settings_email_from_address', __( 'Email From: Address' ) )
		    ->set_default_value( get_option( 'admin_email' ) );
		$tab->addField( 'checkbox', 'crb_settings_email_enable_dev_relay', __( 'Enable dev mail relay' ) )
		    ->set_default_value( EBS_COMMON_DEV_MODE );
		$tab->addField( 'text', 'crb_settings_email_dev_relay_host', __( 'Dev mail relay host' ) )
		    ->set_default_value( 'localhost' )
		    ->set_conditional_logic( [
			    [
				    'field' => 'crb_settings_email_enable_dev_relay',
				    'value' => true,
			    ],
		    ] );
		$tab->addField( 'text', 'crb_settings_email_dev_relay_port', __( 'Dev mail relay port' ) )
		    ->set_default_value( '2500' )
		    ->set_conditional_logic( [
			    [
				    'field' => 'crb_settings_email_enable_dev_relay',
				    'value' => true,
			    ],
		    ] );
	}

	/**
	 *
	 */
	public function registerCarbonFields() {
		$this->createSettingsContainer();

		do_action( $this->getPluginSlug( 'register_fields' ) );
	}

	protected function createSettingsContainer() {
		$container = Container::make( 'theme_options', 'Settings' )
		                      ->set_page_parent( $this->getComponent('adminMenus')->getMenuSlug() );

		$builder = SettingsBuilder::make();

		do_action( $this->getPluginSlug( 'register_settings' ), $builder );

		$builder->apply( $container );
	}

	/**
	 *
	 */
	public function loadCarbonFields() {
		Carbon_Fields::boot();
	}

	/**
	 *
	 */
	public function startSession() {
		if ( ! session_id() && ! headers_sent() ) {
			session_start();
		}
	}

	/**
	 *
	 */
	public function endSession() {
		session_destroy();
	}

	/**
	 *
	 */
	public function run() {
		add_action( 'after_setup_theme', [ $this, 'loadCarbonFields', ] );

		add_action( 'admin_notices', [ $this, 'renderNotices' ] );

		add_action( 'carbon_fields_register_fields', [ $this, 'registerCarbonFields' ] );

		$this->init();
	}

	/**
	 * Init
	 *
	 */
	public function init() {
		add_action( 'init', [ $this, 'wp_init' ] );
		add_action( 'after_setup_theme', [ $this, 'startSession' ], 0 );
		add_action( 'wp_logout', [ $this, 'endSession' ] );
		add_action( 'wp_login', [ $this, 'endSession' ] );

		if ( EBS_COMMON_DEV_MODE ) {
			add_action( 'admin_bar_menu', [ $this, 'add_dev_adminbar_indicator' ], 3, 1 );
			add_action( 'wp_head', [ $this, 'add_dev_adminbar_css' ], 100 );
			add_action( 'admin_head', [ $this, 'add_dev_adminbar_css' ], 100 );
		}

		add_action( 'get_header',
			function () {
				$state = get_option( '_crb_enable_maintenance_mode' );
				if ( $state == 'yes' && ( ! current_user_can( 'manage_options' ) || ! is_user_logged_in() ) ) {
					$logo = "";

					if ( function_exists( 'et_get_option' ) ) {
						$logo = ( $user_logo = et_get_option( 'divi_logo' ) );
					}

					$context = [
						'site_logo' => $logo,
					];

					echo TemplateLoader::render( 'public/maintenance-mode.html.twig', $context );
					exit;
				}
			} );

		add_action( 'phpmailer_init',
			function ( $mailer ) {
				$from    = carbon_get_theme_option( 'crb_settings_email_from_address' );
				$doRelay = carbon_get_theme_option( 'crb_settings_email_enable_dev_relay' );

				$mailer->setFrom( $from );

				if ( $doRelay ) {
					$mailer->isSMTP();
					$mailer->Host = carbon_get_theme_option( 'crb_settings_email_dev_relay_host' );
					$mailer->Port = carbon_get_theme_option( 'crb_settings_email_dev_relay_port' );
				}
			},
			10,
			1 );

		add_filter( 'register_admin_menu_items', [ $this, 'registerAdminMenuItems' ], 10, 1 );
		add_filter( 'admin_menu_order_dashboard', [ $this, 'dashboardMenuOrder' ] );
		add_filter( 'admin_menu_order_crb_carbon_fields_container_settings.php', [ $this, 'settingsMenuOrder' ] );
		add_filter( $this->getPluginSlug( 'register_settings' ), [ $this, 'addSettings' ], 1, 1 );
		add_filter( $this->getPluginSlug( 'update_plugins' ), [ $this, 'registerUpdateCheck' ] );

		TemplateLoader::addTemplatePath( $this->plugin_path . 'templates' );
		TemplateLoader::setContext( [
			'PLUGIN' => [
				'slug'     => $this->name,
				'base_url' => plugin_dir_url( $this->plugin_file ),
			],
		] );

		do_action( $this->getPluginSlug( 'init' ) );
	}

	/**
	 * [ACTION] Stuff to do on WP init
	 *
	 * @see init
	 */
	public function wp_init() {
		$this->createPluginDirs();
	}

	protected function createPluginDirs() {
		foreach ( $this->plugin_dirs as $dir ) {
			if ( $dir['create_if_missing'] && ! file_exists( $dir['path'] ) ) {
				self::get_logger()
				    ->debug( "PluginDir '{$dir['path']}' does not exist, creating" );
				mkdir( $dir['path'] );
			}
		}
	}

	public function get_logger( $name = null ) {
		if ( is_null( $name ) ) {
			$name = $this->getName();
		}

		if ( array_key_exists( $name, self::$loggers ) ) {
			return self::$loggers[ $name ];
		}
		if ( empty( self::$log_handlers ) ) {
			try {
				self::$log_handlers[] = new StreamHandler( $this->getPluginDir( 'data' ) . '/' . $this->name . '_log.txt' );
			} catch ( Exception $e ) {
				error_log( "Failed to open logfile: {$e->getMessage()}" );
			}
		}
		self::$loggers[ $name ] = new Logger( $name );
		foreach ( self::$log_handlers as $handler ) {
			self::$loggers[ $name ]->pushHandler( $handler, Logger::DEBUG );
		}

		return self::$loggers[ $name ];
	}

	/**
	 * Retrieve a PluginDir
	 *
	 * @param string $name
	 *
	 * @return string|null
	 */
	public function getPluginDir( $name ) {
		if ( ! array_key_exists( $name, $this->plugin_dirs ) ) {
			return null;
		}

		return $this->plugin_dirs[ $name ]['path'];
	}

	/**
	 *
	 */
	public function add_dev_adminbar_css() {
		echo '<style type="text/css">' .
		     TemplateLoader::render( 'adminbar-inline-css.css', [] ) .
		     '</style>';
	}
}
