<?php

namespace EbsCommon\Wordpress\Modules;

/**
 * FAQs module
 *
 * @link       https://everbearingservices.com
 * @since      1.0.0
 *
 * @package    EbsCommon
 * @subpackage EbsCommon/includes/components/public
 */

use Carbon_Fields\Container;
use Carbon_Fields\Field;
use EbsCommon\Wordpress\Component;
use EbsCommon\Wordpress\TemplateLoader;

/**
 * FAQs module
 *
 * @package    EbsCommon
 * @author     Everbearing Services <info@everbearingservices.com>
 */
class FaqsComponent extends Component {
	protected $faqs;

	public function __construct( $plugin ) {
		parent::__construct( $plugin );
	}

	/**
	 *
	 * Module initialization
	 *
	 * @since 1.0.0
	 */
	protected function init() {
		add_action( 'wp_enqueue_scripts',
			function () {
				wp_register_script( $this->plugin->getPluginSlug( 'faqs' ),
					plugin_dir_url( __FILE__ ) . 'ebs-common-faqs.js',
					[ 'jquery' ],
					$this->plugin->getVersion(),
					true );

				wp_register_style( $this->plugin->getPluginSlug( 'faqs' ),
					plugin_dir_url( __FILE__ ) . 'ebs-common-faqs.css',
					[],
					$this->plugin->getVersion(),
					'all' );
			} );

		add_filter( 'carbon_fields_register_fields',
			function () {
				$this->register_faqs_menu();
			} );

		add_shortcode( 'faqs',
			function ( $atts ) {
				$atts = shortcode_atts( [
					'format' => 'columns',
				],
					$atts,
					'faqs' );

				if ( is_null( $this->faqs ) ) {
					$this->faqs = carbon_get_theme_option( 'crb_questions' );
				}

				$schemaData = [
					"@context"   => "https://schema.org",
					"@type"      => "FAQPage",
					"mainEntity" => [],
				];

				foreach ( $this->faqs as &$faq ) {
					$faq['answer'] = do_shortcode( $faq['answer'], false );

					$schemaData['mainEntity'][] = [
						'@type'          => 'Question',
						'name'           => $faq['question'],
						'acceptedAnswer' => [
							'@type' => 'Answer',
							'text'  => $faq['answer'],
						],
					];
				}

				return TemplateLoader::render( 'public/faqs.html.twig',
					[
						'format'     => $atts['format'],
						'faqs'       => $this->faqs,
						'schemaJson' => json_encode( $schemaData ),
					] );
			} );
	}

	protected function register_faqs_menu() {
		$parentSlug = $this->plugin->getComponent('adminMenus')->getMenuSlug();

		Container::make( 'theme_options', 'FAQs' )
		         ->set_page_parent( $parentSlug )
		         ->add_fields( [
			         Field::make( 'complex', 'crb_questions', __( 'Frequently Asked Questions', 'lpo-custom' ) )
			              ->set_layout( 'tabbed-vertical' )
			              ->add_fields( [
				              Field::make( 'text', 'question', 'Question' )
				                   ->set_required( true )
				                   ->set_help_text( 'Titles do not need to be UPCASED.  This is done automatically when displayed' ),
				              Field::make( 'text', 'slug', 'Slug' )
				                   ->set_required( true )
				                   ->set_help_text( 'This is used to generate named Anchor tag to allow # addressing in URLs' ),
				              Field::make( 'textarea', 'answer', 'Answer' )
				                   ->set_required( true )
				                   ->set_help_text( 'Use HTML tags for formatting' ),
			              ] )
			              ->set_help_text( 'Tab order determines display order' )
			              ->set_header_template( '
			              <% if (question) { %>
			                <%- question %>
			                <% } %>
			              ' ),
		         ] );
	}

}
