<?php

namespace EbsCommon\Wordpress;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ServerException;
use stdClass;

/**
 * @since       1.0.0
 * @package     EbsCommon
 * @author      Everbearing Services <info@everbearingservices.com>
 */
class Updater extends Component {
	/**
	 * Module initialization
	 *
	 * Register hooks used by module
	 *
	 * @since   1.0.0
	 */
	protected function init() {
		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'check_plugin_updates' ], 10, 1 );
		add_filter( 'pre_set_site_transient_update_themes', [ $this, 'check_theme_updates' ], 10, 1 );
	}

	/**
	 * @param $transient
	 *
	 * @return mixed
	 */
	public function check_theme_updates( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$check_items = apply_filters( $this->plugin->getPluginSlug( 'update_themes' ), [] );
		foreach ( $check_items as $i => $item ) {
			$data = $this->do_update_check_request( $item['url'] );

			if ( ! $data ) {
				continue;
			}

			$obj = [
				'theme'       => $item['name'],
				'new_version' => $data->version,
				'url'         => stripslashes( $data->download_url ),
				'package'     => stripslashes( $data->download_url ),
			];

			if ( version_compare( $item['version'], $obj['new_version'], '<' ) ) {
				$transient->response[ $obj['theme'] ] = $obj;
			} else {
				$transient->last_checked             = time();
				$transient->checked[ $obj['theme'] ] = $item['version'];
			}
		}

		return $transient;
	}

	/**
	 * @param $url
	 *
	 * @return array|mixed|object
	 */
	protected function do_update_check_request( $url ) {
		$client = new Client();
		$res    = null;

		try {
			$res = $client->request( 'GET', $url );

		} catch ( ClientException $e ) {
			if ( $e->hasResponse() && $e->getResponse()->getStatusCode() == 404 ) {
				return [];
			}

			$this->plugin->addFlashNotice( 'error', [ 'text' => $e->getMessage() ] );

			return [];
		} catch ( ServerException $e ) {
			if ( $e->hasResponse() && $e->getResponse()->getStatusCode() == 404 ) {
				return [];
			}

			$this->plugin->addFlashNotice( 'error', [ 'text' => $e->getMessage() ] );

			return [];
		} catch ( ConnectException $e ) {
			$this->plugin->addFlashNotice( 'error', [ 'text' => $e->getMessage() ] );

			return [];
		}

		if ( $res == null || $res->getStatusCode() != 200 ) {
			return [];
		}

		return json_decode( $res->getBody() );

	}

	/**
	 * @param $transient
	 *
	 * @return mixed
	 */
	public function check_plugin_updates( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$check_items = apply_filters( $this->plugin->getPluginSlug( 'update_plugins' ), [] );

		$tag = EBS_COMMON_DEV_MODE ? 'dev' : 'release';

		foreach ( $check_items as $item ) {
			$data = $this->do_update_check_request( $item['url'] );
			if ( ! $data ) {
				continue;
			}

			$obj              = new stdClass();
			$obj->slug        = $item['basename'];
			$obj->new_version = $data->version;
			$obj->url         = stripslashes( $data->download_url );
			$obj->package     = stripslashes( $data->download_url );

			if ( version_compare( $item['version'], $obj->new_version, '<' ) ) {
				$transient->response[ $obj->slug ] = $obj;
			} else {
				$transient->last_checked          = time();
				$transient->checked[ $obj->slug ] = $item['version'];
			}
		}

		return $transient;
	}
}
