<?php

namespace EbsCommon\Wordpress;

use Carbon_Fields\Carbon_Fields;
use Carbon_Fields\Container;
use EbsCommon\Wordpress\Modules\FaqsComponent;
use EbsCommon\Wordpress\Modules\SchemaComponent;
use EbsCommon\Wordpress\Modules\TestimonialsComponent;
use Exception;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use WP_Admin_Bar;
use function EbsCommon\is_divi_theme_builder_context;
use function EbsCommon\joinPaths;

/**
 *
 */
define( 'EBS_COMMON_DEV_MODE', strpos( get_home_url(), 'dev.everbearingservices.com' ) !== false );

/**
 * Class Plugin
 * @package EbsCommon\Wordpress
 */
class Plugin {
	public static $log_handlers = [];

	protected static $loggers = [];

	/**
	 * @var array
	 */
	public $admin_notices;
	/**
	 * @var
	 */
	protected $admin;
	protected $adminMenus;
	/**
	 * @var FilesystemAdapter Symfony Cache object
	 */
	protected $cache;
	protected $components;
	/**
	 * @var
	 */
	protected $name;
	/** @var array List of directories used by plugin */
	protected $plugin_dirs;
	/**
	 * @var
	 */
	protected $plugin_file;
	/**
	 * @var string
	 */
	protected $plugin_path;
	/**
	 * @var Updater
	 */
	protected $updater;
	/**
	 * @var
	 */
	protected $version;

	/**
	 * Plugin constructor.
	 *
	 * @param $name
	 * @param $version
	 * @param $file
	 *
	 * @throws ArgumentError
	 */
	public function __construct( $name, $version, $file ) {
		$this->name          = $name;
		$this->version       = $version;
		$this->plugin_file   = $file;
		$this->plugin_path   = plugin_dir_path( $this->plugin_file );
		$this->admin_notices = [];

		$this->components = [];

		$this->registerComponent( 'adminMenus', AdminMenus::class, [] );
		$this->registerComponent( 'updater', Updater::class, [] );

		if ( $this->enableSchemaModule() ) {
			$this->registerComponent( 'schema', SchemaComponent::class, [] );
		}

		if ( $this->enableFaqsModule() ) {
			$this->registerComponent( 'faqs', FaqsComponent::class, [] );
		}

		if ( $this->enableTestimonialsModule() ) {
			$this->registerComponent( 'testimonials', TestimonialsComponent::class, [] );
		}

		$this->registerSEOPageTitleComponents();

		$this->addPluginDir( 'plugin', plugin_dir_path( $this->plugin_file ), false )
		     ->addPluginDir( 'data', wp_get_upload_dir()['basedir'] . '/' . $this->name, true )
		     ->addPluginDir( 'cache_base', ABSPATH . wp_basename( content_url() ) . '/cache/', true )
		     ->addPluginSubdir( 'cache_base', 'cache', $this->name, true );


		$this->cache = new FilesystemAdapter( 'cache', 0, ABSPATH . wp_basename( content_url() ) . '/cache/' . $name );
	}

	public function registerComponent( $name, $class, $deps ) {
		$this->components[ $name ] = [
			'class'    => $class,
			'deps'     => $deps,
			'instance' => new $class( $this ),
		];
	}

	protected function registerSEOPageTitleComponents() {
		add_action( 'init', function () {
			add_action( 'add_meta_boxes', function () {
				add_meta_box(
					'seo_page_title_meta_box', // ID of the meta box
					'SEO Customization', // Title of the meta box
					[ $this, 'renderSEOTitleMetaBox' ],
					'page', // Post type
					'normal', // Context
					'high' // Priority
				);
			} );
			add_action( 'save_post', [ $this, 'saveSEOTitleMetaBox' ] );

			add_filter( 'et_builder_resolve_dynamic_content_dynamic_page_title',
				function ( $content, $settings, $postID, $context, $overrides ) {
					if ( is_divi_theme_builder_context() ) {
						return '<h1> Dynamic Page Title</h1>';
					}

					// Check if we're inside a post or page
					if ( isset( $postID ) ) {
						// Attempt to retrieve the value of the 'seo_page_title' meta key
						$seo_page_title = get_post_meta( $postID, 'seo_page_title', true );

						// Check if 'seo_page_title' has a value
						if ( ! empty( $seo_page_title ) ) {
							// If 'seo_page_title' exists and has a value, return it
							return '<h1>' . $seo_page_title . '</h1>';
						} else {
							// If 'seo_page_title' does not have a value, return the natural page title
							return '<h1>' . get_the_title( $postID ) . '</h1>';
						}
					}

					// Fallback in case we're not inside a post or page context
					return 'Not a post or page';
				},
				100,
				6 );

			add_filter( 'et_builder_custom_dynamic_content_fields', function ( $fields ) {
				$fields['dynamic_page_title'] = [
					'label'           => 'Dynamic Page Title',
					'type'            => 'text',
					'option_category' => 'basic_option',
					'description'     => 'Insert either natural page title, or one set w/ seo_page_title meta',
				];

				return $fields;
			} );

			add_filter( 'document_title_parts', function ( $title ) {
				// Access the global post object to get the current post's ID
				global $post;
				if ( is_null( $post ) ) {
					return $title;
				}
				$yoast_title = get_post_meta( $post->ID, '_yoast_wpseo_title', true );

				// Check if we're inside a post or page and the seo_page_title meta key exists
				if ( ! $yoast_title && is_singular() && isset( $post->ID ) && $seo_title = get_post_meta( $post->ID,
						'seo_page_title',
						true ) ) {
					// If 'seo_page_title' has a value, override the title
					if ( ! empty( $seo_title ) ) {
						$title['title'] = $seo_title;
					}
				}

				return $title;
			} );

			add_filter( 'wpseo_title', function ( $title ) {
				// Access the global post object to get the current post's ID
				global $post;
				if ( is_null( $post ) ) {
					return $title;
				}
				$yoast_title = get_post_meta( $post->ID, '_yoast_wpseo_title', true );

				// Ensure we are on a singular post or page and the global post object is available
				if ( ! $yoast_title && is_singular() && isset( $post->ID ) ) {
					// Get the natural page title
					$natural_page_title = get_the_title( $post->ID );

					// Attempt to retrieve the value of the 'seo_page_title' meta key
					$seo_page_title = get_post_meta( $post->ID, 'seo_page_title', true );

					// Check if 'seo_page_title' has a value and is different from the natural page title
					if ( ! empty( $seo_page_title ) && $seo_page_title !== $natural_page_title ) {
						// Replace only the first occurrence of the natural page title in the Yoast title
						$pos = strpos( $title, $natural_page_title );
						if ( $pos !== false ) {
							return substr_replace( $title, $seo_page_title, $pos, strlen( $natural_page_title ) );
						}
					}
				}

				// Return the default Yoast SEO title if no custom 'seo_page_title' is set, or other conditions aren't met
				return $title;
			} );
		} );
	}

	public function renderSEOTitleMetaBox( $post ) {
		// Use nonce for verification
		wp_nonce_field( 'seo_page_title_meta_box_nonce', 'seo_page_title_meta_box_nonce_field' );

		// Get the current value if it exists
		$seo_page_title = get_post_meta( $post->ID, 'seo_page_title', true );

		// Meta box HTML
		echo '<label for="seo_page_title_field">SEO Page Title: </label>';
		echo '<input type="text" id="seo_page_title_field" name="seo_page_title_field" value="' . esc_attr( $seo_page_title ) . '" size="25" />';
	}

	public function saveSEOTitleMetaBox( $post_id ) {
		// Check for nonce security
		if ( ! isset( $_POST['seo_page_title_meta_box_nonce_field'] ) || ! wp_verify_nonce( $_POST['seo_page_title_meta_box_nonce_field'],
				'seo_page_title_meta_box_nonce' ) ) {
			return;
		}

		// Prevent autosave
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check user permissions
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save the meta box data
		if ( isset( $_POST['seo_page_title_field'] ) ) {
			update_post_meta( $post_id, 'seo_page_title', sanitize_text_field( $_POST['seo_page_title_field'] ) );
		}
	}

	protected function enableSchemaModule() {
		return true;
	}

	protected function enableFaqsModule() {
		return true;
	}

	protected function enableTestimonialsModule() {
		return true;
	}

	/**
	 * Add a subdirectory to another plugin directory
	 *
	 * @param string $parent            Parent PluginDir to add subdirectory to
	 * @param string $name              Name of subdirectory PluginDir entry
	 * @param string $path              Allows specification of physical subdirectory name, defaults ot $name
	 * @param bool   $create_if_missing Should directory be created if missing?  Default true
	 *
	 * @return $this
	 * @throws ArgumentError
	 *
	 */
	public function addPluginSubdir( $parent, $name, $path = '', $create_if_missing = true ) {
		if ( ! array_key_exists( $parent, $this->plugin_dirs ) ) {
			throw new ArgumentError( "Parent dir {$parent} does not exist" );
		}

		$pd                         = $this->plugin_dirs[ $parent ];
		$this->plugin_dirs[ $name ] = [
			'path'              => joinPaths( $pd['path'], $path ? $path : $name ),
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	/**
	 * Add a PluginDir entry
	 *
	 * @param string $name              Name of PluginDir entry
	 * @param string $path              Path to PluginDir entry
	 * @param bool   $create_if_missing Should the directory be created if missing?  Default true
	 *
	 * @return $this
	 */
	public function addPluginDir( $name, $path, $create_if_missing = true ) {
		$this->plugin_dirs[ $name ] = [
			'path'              => $path,
			'create_if_missing' => $create_if_missing,
		];

		return $this;
	}

	public function cache() {
		return $this->cache;
	}

	/**
	 * Get absolute
	 *
	 * @param $path
	 *
	 * @return string
	 */
	public function getAssetUrl( $path ) {
		return plugin_dir_url( $this->plugin_file ) . 'assets/' . $path;
	}

	/**
	 * @return mixed
	 */
	public function getPluginFile() {
		return $this->plugin_file;
	}

	/**
	 * @param array $items
	 *
	 * @return array
	 */
	public function registerUpdateCheck( $items = [] ) {
		$items[] = [
			'basename' => $this->getUpdateBasename(),
			'name'     => $this->getName(),
			'url'      => $this->getUpdateUrl(),
			'version'  => $this->getVersion(),
		];

		return $items;
	}

	public function getUpdateBasename() {
		return $this->getName() . '/' . basename( $this->plugin_file );
	}

	/**
	 * @return mixed
	 */
	public function getName() {
		return $this->name;
	}

	public function getUpdateUrl() {
		return "https://ebsdev.everbearingservices.com/api/v1/plugin-repos/{$this->getName()}/latest";
	}

	/**
	 * @return mixed
	 */
	public function getVersion() {
		return $this->version;
	}

	public function registerAdminMenuItems( $menu_items ) {
		if ( $this->enableDashboard() ) {
			$menu_items[] = [
				'title'  => 'Dashboard',
				'slug'   => 'dashboard',
				'func',
				[ $this, 'render_dashboard_page' ],
				'parent' => $this->getComponent( 'adminMenus' )->getMenuSlug(),
			];
		}

		return $menu_items;
	}

	/**
	 * Should the Dasbhoard module be enabled
	 *
	 * @return bool
	 */
	public function enableDashboard() {
		return true;
	}

	public function getComponent( $name ) {
		if ( isset( $this->components[ $name ] ) ) {
			return $this->components[ $name ]['instance'];
		}

		return null;
	}

	/**
	 * Specify position of Top llevel menu in WP Admin menu sidebar
	 *
	 * @return int Top level menu position
	 */
	public function topLevelMenuOrder() {
		return 2;
	}

	/**
	 * @return int
	 */
	public function dashboardMenuOrder() {
		return - 1;
	}

	/**
	 * Sort order for Settings menu item
	 *
	 * @return int Relative order of Settings
	 */
	public function settingsMenuOrder() {
		return 99999;
	}

	/**
	 * @param $level
	 * @param $args
	 */
	public function addAdminNotice( $level, $args ) {
		$args['level'] = $level;

		$this->admin_notices[] = $args;
	}

	/**
	 * Add flash notice for a User
	 *
	 * Defaults to current user
	 *
	 * @param string $level   Notice level, [warning|info|success|error]
	 * @param array  $args    Flash notice arguments
	 * @param int    $user_id User ID to add message for
	 *
	 * @return $this
	 */
	public function addFlashNotice( $level, $args, $user_id = null ) {
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}
		$args['level'] = $level;

		update_user_meta( $user_id, $this->getPluginSlug( 'flash_message_' . microtime() ), $args );

		return $this;
	}

	/**
	 * @param        $slug
	 * @param string $sep
	 *
	 * @return string
	 */
	public function getPluginSlug( $slug, $sep = '-' ) {
		return $this->name . $sep . $slug;
	}

	/**
	 *
	 */
	public function renderNotices() {
		$ns = array_map( function ( $x ) {
			return $this->configNotice( $x );
		},
			$this->admin_notices );

		$flash_notices = [];
		$user          = get_current_user_id();
		$metadata      = get_user_meta( $user );

		$slug = $this->getPluginSlug( 'flash_message' );
		foreach ( $metadata as $key => $value ) {
			$t = substr( $key, 0, strlen( $slug ) ) == $slug;
			if ( $t ) {
				$flash_notices[] = $this->configNotice( unserialize( $value[0] ) );
				delete_user_meta( $user, $key );
			}
		}

		echo TemplateLoader::render( 'notices.html.twig',
			[
				'plugin_slug'   => $this->getName(),
				'admin_notices' => $ns,
				'flash_notices' => $flash_notices,
			] );
	}

	/**
	 * @param $n
	 *
	 * @return mixed
	 */
	public function configNotice( $n ) {
		if ( ! isset( $n['content'] ) ) {
			$n['content'] = '';
		}

		if ( isset( $n['text'] ) ) {
			$n['content'] = '<p>' . $n['text'] . '</p>';
		}

		if ( isset( $n['cb'] ) ) {
			if ( is_callable( $n['cb'] ) ) {
				$n['content'] = $n['cb']();
			}
		}

		return $n;
	}

	/**
	 * @param SettingsBuilder $builder
	 */
	public function addSettings( $builder ) {
		$this->registerGeneralSettings( $builder );
		$this->registerEmailSettings( $builder );
	}

	protected function registerGeneralSettings( $builder ) {
		$tab = $builder->addTab( 'General', - 10 );
		$tab->addField( 'text', 'crb_password_strength', __( 'Password Strength', 'lpo-custom' ) )
		    ->set_default_value( 2 )
		    ->set_required( true );
		$tab->addField( 'textarea', 'crb_password_hint', __( 'Password hint', 'lpo-custom' ) )
		    ->set_default_value( 'Hint: The password should be at least twelve characters long. To make it stronger, use upper and lower case letters, numbers, and symbols like ! " ? $ % ^ &amp; ).' )
		    ->set_required( true );
		$tab->addField( 'checkbox', 'crb_enable_maintenance_mode', __( 'Enable maintenance mode' ) )
		    ->set_default_value( false );
		$tab->addField( 'checkbox', 'crb_enable_development_mode', __( 'Enable Development mode' ) )
		    ->set_default_value( false )
		    ->set_help_text( 'Development mode will prevent public access' );
		$tab->addField( 'textarea', 'crb_development_mode_whitelist', __( 'Development mode whitelisted IPs' ) )
		    ->set_help_text( 'These IPs can access the site w/o needing to be logged in' );

	}

	protected function registerEmailSettings( $builder ) {
		$tab = $builder->addTab( 'Email', 8 );
		$tab->addField( 'text', 'crb_settings_email_from_address', __( 'Email From: Address' ) )
		    ->set_default_value( get_option( 'admin_email' ) );
		$tab->addField( 'text', 'crb_settings_email_from_name', __( 'Email From Name' ) );
		$tab->addField( 'text', 'crb_settings_email_return_path', __( 'Email Return Path' ) );
		$tab->addField( 'checkbox', 'crb_settings_email_enable_dev_relay', __( 'Enable dev mail relay' ) )
		    ->set_default_value( EBS_COMMON_DEV_MODE );
		$tab->addField( 'text', 'crb_settings_email_dev_relay_host', __( 'Dev mail relay host' ) )
		    ->set_default_value( 'localhost' )
		    ->set_conditional_logic( [
			    [
				    'field' => 'crb_settings_email_enable_dev_relay',
				    'value' => true,
			    ],
		    ] );
		$tab->addField( 'text', 'crb_settings_email_dev_relay_port', __( 'Dev mail relay port' ) )
		    ->set_default_value( '2500' )
		    ->set_conditional_logic( [
			    [
				    'field' => 'crb_settings_email_enable_dev_relay',
				    'value' => true,
			    ],
		    ] );
	}

	/**
	 *
	 */
	public function registerCarbonFields() {
		if ( $this->enableSettings() ) {
			$this->createSettingsContainer();
		}

		do_action( $this->getPluginSlug( 'register_fields' ) );
	}

	/**
	 * @return bool
	 */
	public function enableSettings() {
		return true;
	}

	protected function createSettingsContainer() {
		$container = Container::make( 'theme_options', 'Settings' )
		                      ->set_page_parent( $this->getComponent( 'adminMenus' )->getMenuSlug() );

		$builder = SettingsBuilder::make();

		do_action( $this->getPluginSlug( 'register_settings' ), $builder );
		$this->registerSettings( $builder );

		$builder->apply( $container );
	}

	protected function registerSettings( $builder ) {
	}

	/**
	 *
	 */
	public function loadCarbonFields() {
		Carbon_Fields::boot();
	}

	/**
	 *
	 */
	public function startSession() {
		if ( ! session_id() && ! headers_sent() ) {
			session_start();
		}
	}

	/**
	 *
	 */
	public function endSession() {
		session_destroy();
	}

	/**
	 *
	 */
	public function run() {
		add_action( 'after_setup_theme', [ $this, 'loadCarbonFields', ] );

		add_action( 'admin_notices', [ $this, 'renderNotices' ] );

		add_action( 'carbon_fields_register_fields', [ $this, 'registerCarbonFields' ] );

		$this->init();
	}

	/**
	 * Init
	 *
	 */
	public function init() {
		add_action( 'init', [ $this, 'wp_init' ] );

		if ( $this->enableAdminMenu() ) {
			add_action( 'admin_bar_menu',
				/**
				 * @param WP_Admin_Bar $adminbar
				 *
				 * @return mixed
				 */
				function ( $adminbar ) {
					$adminbar->add_node( [
						'id'    => 'ebs-common',
						'title' => "v{$this->version}",
						'href=' => '#',
					] );

					return $adminbar;
				},
				3,
				1 );
		}
		if ( EBS_COMMON_DEV_MODE ) {
			add_action( 'wp_head', [ $this, 'add_dev_adminbar_css' ], 100 );
			add_action( 'admin_head', [ $this, 'add_dev_adminbar_css' ], 100 );
			add_filter( 'option_blogname',
				function ( $value, $option ) {
					return $value . " [DEV]";
				},
				10,
				2 );
			add_filter( 'admin_title', function ( $title ) {
				return "[DEV] $title";
			}, 1000 );
		}

		if ( $this->enableSettings() ) {
			add_filter( 'admin_menu_order_crb_carbon_fields_container_settings.php', [ $this, 'settingsMenuOrder' ] );
			add_filter( $this->getPluginSlug( 'register_settings' ), [ $this, 'addSettings' ], 1, 1 );

			add_action( 'get_header',
				function () {
					$state     = carbon_get_theme_option( 'crb_enable_development_mode' );
					$whitelist = carbon_get_theme_option( 'crb_development_mode_whitelist' );

					if ( $state == 'yes' && ( ! current_user_can( 'manage_options' ) || ! is_user_logged_in() ) ) {
						$logo = "";

						if ( function_exists( 'et_get_option' ) ) {
							$logo = ( $user_logo = et_get_option( 'divi_logo' ) );
						}

						$context = [
							'site_logo' => $logo,
						];

						echo TemplateLoader::render( 'public/development-mode.html.twig', $context );
						exit;
					}

					$state = get_option( '_crb_enable_maintenance_mode' );
					if ( $state == 'yes' && ( ! current_user_can( 'manage_options' ) || ! is_user_logged_in() ) ) {
						$logo = "";

						if ( function_exists( 'et_get_option' ) ) {
							$logo = ( $user_logo = et_get_option( 'divi_logo' ) );
						}

						$context = [
							'site_logo' => $logo,
						];

						echo TemplateLoader::render( 'public/maintenance-mode.html.twig', $context );
						exit;
					}

				} );

			add_action( 'phpmailer_init',
				function ( $mailer ) {
					$from       = carbon_get_theme_option( 'crb_settings_email_from_address' );
					$fromName   = carbon_get_theme_option( 'crb_settings_email_from_name' );
					$returnPath = carbon_get_theme_option( 'crb_settings_email_return_path' );

					$doRelay = carbon_get_theme_option( 'crb_settings_email_enable_dev_relay' );

					if ( $returnPath ) {
						$mailer->Sender = $returnPath;
					}

					if ( $from && $fromName ) {
						$mailer->setFrom( $from, $fromName );
					}

					if ( $doRelay ) {
						$mailer->isSMTP();
						$mailer->Host = carbon_get_theme_option( 'crb_settings_email_dev_relay_host' );
						$mailer->Port = carbon_get_theme_option( 'crb_settings_email_dev_relay_port' );
					}
				},
				10,
				1 );
		}

		add_filter( 'register_admin_menu_items', [ $this, 'registerAdminMenuItems' ], 10, 1 );

		add_filter( $this->getPluginSlug( 'update_plugins' ), [ $this, 'registerUpdateCheck' ] );

		$this->registerTemplatePaths();
		TemplateLoader::setContext( [
			'PLUGIN' => [
				'slug'     => $this->name,
				'base_url' => plugin_dir_url( $this->plugin_file ),
			],
		] );

		do_action( $this->getPluginSlug( 'init' ) );
	}

	protected function registerTemplatePaths() {
		// Check if theme path exists
		if ( file_exists( get_stylesheet_directory() . '/templates' ) ) {
			TemplateLoader::addTemplatePath( get_stylesheet_directory() . '/templates' );
		}

		TemplateLoader::addTemplatePath( $this->plugin_path . 'templates' );
		TemplateLoader::addTemplatePath( __DIR__ . '/../templates' );
	}

	/**
	 * Control whether the admin menu dropdown should be added
	 *
	 * Default: True
	 *
	 * @return bool
	 */
	public function enableAdminMenu() {
		return true;
	}

	/**
	 * Code to run on plugin activation
	 */
	public function onActivation() {
	}

	/**
	 * Code to run on plugin deactivation
	 */
	public function onDeactivation() {
	}

	/**
	 * [ACTION] Stuff to do on WP init
	 *
	 * @see init
	 */
	public function wp_init() {
		$this->createPluginDirs();
	}

	protected function createPluginDirs() {
		foreach ( $this->plugin_dirs as $dir ) {
			if ( $dir['create_if_missing'] && ! file_exists( $dir['path'] ) ) {
				self::get_logger()
				    ->debug( "PluginDir '{$dir['path']}' does not exist, creating" );
				mkdir( $dir['path'] );
			}
		}
	}

	public function get_logger( $name = null, $level = Logger::INFO ) {
		if ( is_null( $name ) ) {
			$name = $this->getName();
		}

		if ( array_key_exists( $name, self::$loggers ) ) {
			return self::$loggers[ $name ];
		}
		if ( empty( self::$log_handlers ) ) {
			try {
				self::$log_handlers[] = new StreamHandler( $this->getPluginDir( 'data' ) . '/' . $this->name . '_log.txt',
					$level );
			} catch ( Exception $e ) {
				error_log( "Failed to open logfile: {$e->getMessage()}" );
			}
		}
		self::$loggers[ $name ] = new Logger( $name );
		foreach ( self::$log_handlers as $handler ) {
			self::$loggers[ $name ]->pushHandler( $handler );
		}

		return self::$loggers[ $name ];
	}

	/**
	 * Retrieve a PluginDir
	 *
	 * @param string $name
	 *
	 * @return string|null
	 */
	public function getPluginDir( $name ) {
		if ( ! array_key_exists( $name, $this->plugin_dirs ) ) {
			return null;
		}

		return $this->plugin_dirs[ $name ]['path'];
	}

	/**
	 *
	 */
	public function add_dev_adminbar_css() {
		echo '<style type="text/css">' .
		     TemplateLoader::render( 'adminbar-inline-css.css', [] ) .
		     '</style>';
	}
}
