<?php

namespace EbsCommon\Wordpress\Modules;

/**
 * Schema.org support module
 *
 * @link       https://everbearingservices.com
 * @since      1.1.0
 *
 * @package    EbsCommon
 * @subpackage EbsCommon/includes/modules
 */

use Carbon_Fields\Container;
use Carbon_Fields\Field;
use EbsCommon\Wordpress\Component;
use function EbsCommon\addIf;

/**
 * Schema.org support module
 *
 * @package    EbsCommon
 * @author     Everbearing Services <info@everbearingservices.com>
 */
class SchemaComponent extends Component {
	protected static $daysOfWeek = [
		'sunday'    => 'Sunday',
		'monday'    => 'Monday',
		'tuesday'   => 'Tuesday',
		'wednesday' => 'Wednesday',
		'thursday'  => 'Thursday',
		'friday'    => 'Friday',
		'saturday'  => 'Saturday',
	];

	/**
	 *
	 * Module initialization
	 *
	 * @since 1.0.0
	 */
	protected function init() {
		add_filter( 'carbon_fields_register_fields',
			function () {
				$this->register_schema_menu();
			} );

		add_action( 'wp_head',
			function () {
				$schemaData = $this->buildSchemaData();
				if ( $schemaData ) {
					echo '<script type="application/ld+json">' . json_encode( $schemaData ) . '</script>';
				}
			} );
	}

	protected function register_schema_menu() {
		if ( function_exists( 'et_get_option' ) ) {
			$logo = et_get_option( 'divi_logo' );
		} else {
			$logo = get_custom_logo();
		}

		$parentSlug = $this->plugin->getComponent( 'adminMenus' )->getMenuSlug();

		Container::make( 'theme_options', 'Schema' )
		         ->set_page_parent( $parentSlug )
		         ->add_fields( [
			         Field::make( 'complex', 'crb_structured_data' )
			              ->add_fields( [
				              Field::make( 'text', 'name', 'Name' )
				                   ->set_required( true )
				                   ->set_default_value( get_bloginfo() ),
				              Field::make( 'image', 'logo', 'Logo' )
				                   ->set_required( true )
				                   ->set_value_type( 'url' )
				                   ->set_default_value( $logo ),
				              Field::make( 'text', 'id', 'ID' )
				                   ->set_required( true )
				                   ->set_default_value( get_bloginfo( 'wpurl' ) . '#' )
				                   ->set_help_text( 'Globally unique ID of the specific business location in the form of a URL. The ID should be stable and unchanging over time. Google Search treats the URL as an opaque string and it does not have to be a working link.' ),
				              Field::make( 'text', 'url', 'URL' )
				                   ->set_required( true )
				                   ->set_default_value( get_bloginfo( 'wpurl' ) ),
				              Field::make( 'text', 'phone', 'Phone' ),
				              Field::make( 'text', 'street', 'Street' ),
				              Field::make( 'text', 'city', 'City' ),
				              Field::make( 'text', 'zipcode', 'Zip Code' ),
				              Field::make( 'text', 'country', 'Country' ),
				              Field::make( 'text', 'state', 'State' ),
				              Field::make( 'text', 'latitude', 'Latitude' ),
				              Field::make( 'text', 'longitude', 'Longitude' ),
				              Field::make( 'text', 'price-range', 'Price Range' ),
				              Field::make( 'text', 'currency-accepted', 'Currency Accepted' )
				                   ->set_default_value( 'USD' ),
				              Field::make( 'checkbox', 'always-open', 'Open 24/7' )
				                   ->set_default_value( false ),
				              Field::make( 'complex', 'hours', 'Hours of Operation' )
				                   ->set_conditional_logic( [
					                   [
						                   'field'    => 'always-open',
						                   'value'    => false,
						                   'comppare' => '=',
					                   ],
				                   ] )
				                   ->add_fields( [
					                   Field::make( 'multiselect', 'days', 'Day(s)' )
					                        ->set_options( self::$daysOfWeek ),
					                   Field::make( 'text', 'opens', 'Open' ),
					                   Field::make( 'text', 'closes', 'Close' ),
				                   ] )
				                   ->set_header_template( '
										<% if (days) { %>
										<- days >
										<% } %>
									' ),
			              ] ),
		         ] );
	}

	protected function buildSchemaData() {
		$config  = carbon_get_theme_option( 'crb_structured_data' );
		$schemas = [];

		foreach ( $config as $item ) {
			$schemaData = [
				'@context'                  => 'https://schema.org',
				'@type'                     => 'LocalBusiness',
				'name'                      => $item['name'],
				'image'                     => $item['logo'],
				'@id'                       => $item['id'],
				'url'                       => $item['url'],
				'telephone'                 => $item['phone'],
				'address'                   => [
					'@type'           => 'PostalAddress',
					'streetAddress'   => $item['street'],
					'addressLocality' => $item['city'],
					'addressRegion'   => $item['state'],
					'postalCode'      => $item['zipcode'],
					'addressCountry'  => $item['country'],
				],
				'openingHoursSpecification' => [],
				'priceRange'                => $item['price-range'],
			];

			addIf( $schemaData, 'currenciesAccepted', $item['currency-accepted'] );
			addIf( $schemaData, 'longitude', $item['longitude'] );
			addIf( $schemaData, 'latitude', $item['latitude'] );

			$hours = [];
			if ( $item['always-open'] ) {
				$t = [
					'dayOfWeek' => [],
					'opens'     => '00:00',
					'closes'    => '23:59',
				];

				foreach ( self::$daysOfWeek as $day ) {
					$t['dayOfWeek'][] = $day;
				}

				$hours[] = $t;
			} else {
				foreach ( $item['hours'] as $hourSpec ) {
					$t = [
						'dayOfWeek' => [],
						'opens'     => $hourSpec['opens'],
						'closes'    => $hourSpec['closes'],
					];
					foreach ( $hourSpec['days'] as $day ) {
						$t['dayOfWeek'][] = self::$daysOfWeek[ $day ];
					}

					$hours[] = $t;
				}
			}

			foreach ( $hours as $hourSpec ) {
				$newItem                                   = array_merge(
					[ '@type' => 'OpeningHoursSpecification' ],
					$hourSpec
				);
				$schemaData['openingHoursSpecification'][] = $newItem;
			}

			$schemas[] = $schemaData;
		}

		return $schemas;
	}

}
