const gulp = require('gulp');
const gulpZip = require('gulp-zip');
const log = require('fancy-log');
const fs = require('fs');
const cproc = require('child_process');
const path = require('path');

const plugin_slug = 'ebs-common';
const plugin_version_name = 'EBS_COMMON';
const plugin_main_file = path.join(plugin_slug, plugin_slug + '.php');
const plugin_readme_file = path.join(plugin_slug, 'README.md');

const plugin_meta_version_re = /\*\w* Version:\W*(.*)/gm;
const version_re = new RegExp('\'' + plugin_version_name + '_VERSION\', \'(.*)\'', 'gm');

const args = (argList => {
  let arg = {}, a, opt, thisOpt, curOpt;
  for (a = 0; a < argList.length; a++) {
    thisOpt = argList[a].trim();
    opt = thisOpt.replace(/^\-+/, '');

    if (opt === thisOpt) {
      // argument value
      if (curOpt) {
        arg[curOpt] = opt;
      }
      curOpt = null;
    }
    else {
      // argument name
      curOpt = opt;
      arg[curOpt] = true;
    }
  }
  return arg;
})(process.argv);

function getVersion (fn) {
  let fileText = fs.readFileSync(fn);

  let versions = {
    wpVersion: plugin_meta_version_re.exec(fileText)[ 1 ],
    defVersion: version_re.exec(fileText)[ 1 ],
    isDev: false
  };

  if (versions.wpVersion.indexOf('dev') > -1) {
    versions.isDev = true;
  }

  return versions;
}

function checkReadmeVersion (fn, version) {
  let re = new RegExp('### v' + version + '.*', 'gm');

  return re.exec(fs.readFileSync(fn));
}

function withError (msg) {
  let err = new Error(msg);
  err.showStack = false;
  return err;
}

let version;


gulp.task('setup-version', done => {
  version = getVersion(plugin_main_file);
  done();
});

gulp.task('check-version', done => {
  if (version.wpVersion !== version.defVersion) {
    done(withError('Version mismatch: WP Plugin Block says "' + version.wpVersion + '" but Plugin Defined version says "' + version.defVersion + '"'));
  }

  // Check for version changelog in readme and changelog
  if (!checkReadmeVersion(plugin_readme_file, version.wpVersion)) {
    done(withError('Missing changelog entry in README.md'));
  }

  done();
});

gulp.task('install-deps', done => {
  cproc.execSync('composer install', {
    cwd: plugin_slug + '/'
  });

  done();
});

gulp.task('dump-autoload', done => {
  cproc.execSync('composer dumpautoload', {
    cwd: plugin_slug + '/'
  });

  done();
});

gulp.task('do-build', done => {
  let dest_file = plugin_slug + '-' + version.defVersion + '.zip';

  log('Building version ' + version.defVersion + ' to dist/' + dest_file + '...');

  return gulp
    .src(path.join(plugin_slug, '**/*'), { base: '.' })
    .pipe(gulpZip(dest_file))
    .pipe(gulp.dest('dist/'));
});

gulp.task('default', gulp.series('setup-version', 'check-version', 'install-deps', 'do-build'));

gulp.task('build:dev', gulp.series('setup-version', 'install-deps', 'do-build'));
